/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "SetSynchronizedSurfacesCommand.h"
#include "ICommandExecutor.h"
#include "Scene.h"
#include "Log.h"

ExecutionResult SetSynchronizedSurfacesCommand::execute(ICommandExecutor* executor)
{
    Scene& scene = *(executor->getScene());
    ExecutionResult result = ExecutionFailed;
    IRenderer* renderer = *(executor->getRendererList())->begin();
    unsigned int layerId;

    LOG_DEBUG("SetSynchronizedSurfacesCommand", "Length to set=" << m_length);

    // First of all get all surfaces which are in the current render order.
	if (renderer)
	{
		for (unsigned int i = 0; i < m_length; i++)
		{
	        Surface* surface = scene.getSurface(m_array[i]);
            if (renderer->isSurfaceDirty(m_array[i]))
            {
    	        layerId = surface->getContainingLayerId();
                if (false == areAllScreensFrozen(layerId, executor))
				{
                    surface->ackClearDamage = true;
                    scene.m_CmdFenceWaitCount++;
                    LOG_DEBUG("SetSynchronizedSurfacesCommand",
		                  "surface, ID=" << m_array[i] << " "
		                  "damaged when setting to synchronized "
		                  "fence count =" << scene.m_CmdFenceWaitCount);
                }
                else
                {
                    LOG_DEBUG("SetSynchronizedSurfacesCommand",
		                  "All screens containing surfaceID=" << m_array[i] << "are frozen");
                }
            }
        }
    }
    if (scene.m_CmdFenceWaitCount > 0)
    {
        LOG_DEBUG("SetSynchronizedSurfacesCommand", "Waiting for damage clear");
        scene.unlockScene();
        scene.waitForDamageClear();
        scene.lockScene();
        LOG_DEBUG("SetSynchronizedSurfacesCommand", "Damage is cleared");
    }

    for (unsigned int i = 0; i < m_length; i++)
    {
        Surface* surface = scene.getSurface(m_array[i]);
        if (surface)
        {
        	surface->setSynchronized(true);
            LOG_DEBUG("SetSynchronizedSurfacesCommand",
                      "Setting surface, ID=" << m_array[i] << " "
                      "to synchronized");
            result = ExecutionSuccess;
        }
    }
    return result;
}

const std::string SetSynchronizedSurfacesCommand::getString()
{
    std::stringstream description;
    description << "SetSynchronizedSurfacesCommand("
                << "m_array=[";

    for (unsigned int i = 0; i < m_length; ++i)
    {
        description << m_array[i] << ",";
    }
    description << "], m_length=" << m_length
                << ")";
    return description.str();
}

bool SetSynchronizedSurfacesCommand::
    areAllScreensFrozen(unsigned int layerID, ICommandExecutor* executor)
{
	Scene* pScene = executor->getScene();
	IRenderer* pRenderer = *(executor->getRendererList())->begin();
    unsigned int screenID = 0;
	LmScreenList screenList = pScene->getScreenList();
    LmScreenListIterator iter = screenList.begin();
    LmScreenListIterator iterEnd = screenList.end();
    bool allScreensFrozen = false;

    for (; iter != iterEnd; ++iter)
    {
    	screenID = (*iter)->getID();
        LayerList layers = pScene->getCurrentRenderOrder(screenID);
        if (layers.size() > 0)
        {
            for (LayerListConstIterator layer = layers.begin(); layer != layers.end(); layer++)
            {
            	if(layerID == (*layer)->getID())
            	{
            		if (pRenderer->isScreenFrozen(screenID))
            		{
            		    allScreensFrozen = true;
            		}
            		else
            		{
                        LOG_DEBUG("SetSynchronizedSurfacesCommand",
    		                  "screen, ID=" << screenID << " "
    		                  "containing layerID=" << layerID <<
    		                  ", is not frozen");
            		    allScreensFrozen = false;
            		    break;
            		}
            	}
            }
        }
    }

    return allScreensFrozen;
}
